/*
Boost Software License - Version 1.0 - August 17th, 2003

Permission is hereby granted, free of charge, to any person or organization
obtaining a copy of the software and accompanying documentation covered by
this license (the "Software") to use, reproduce, display, distribute,
execute, and transmit the Software, and to prepare derivative works of the
Software, and to permit third-parties to whom the Software is furnished to
do so, all subject to the following:

The copyright notices in the Software and this entire statement, including
the above license grant, this restriction and the following disclaimer,
must be included in all copies of the Software, in whole or in part, and
all derivative works of the Software, unless such copies or derivative
works are solely in the form of machine-executable object code generated by
a source language processor.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.
*/

#ifndef LIBINTL_INTERNAL_UTIL_HPP_
#define LIBINTL_INTERNAL_UTIL_HPP_

#if defined(WIN32) || defined(WINCE)
typedef unsigned int uint32_t;
#else
#include <stdint.h>
#endif

namespace libintllite
{

namespace internal
{

// Helper functions for handling numbers and char array conversions:

static inline bool isBigEndian()
{
	uint32_t checkNumber = 0x1100;
	return (*reinterpret_cast<const char*>(&checkNumber) != 0);
}

static inline uint32_t swapUInt32Bytes(uint32_t number)
{
	const char* numberAsCharArray = reinterpret_cast<const char*>(&number);

	uint32_t swappedNumber;
	char* swappedNumberAsCharArray = reinterpret_cast<char*>(&swappedNumber);
	swappedNumberAsCharArray[0] = numberAsCharArray[3];
	swappedNumberAsCharArray[1] = numberAsCharArray[2];
	swappedNumberAsCharArray[2] = numberAsCharArray[1];
	swappedNumberAsCharArray[3] = numberAsCharArray[0];
	return swappedNumber;
}

static inline uint32_t charArrayToUInt32(const char uint32CharArray[4])
{
	return *reinterpret_cast<const uint32_t*>(uint32CharArray);
}

static inline bool readUIn32FromFile(FILE* fileHandle, bool needsBeToLeConversion, uint32_t& outReadUInt32)
{
	char uint32CharArray[4];
	if ((fread(uint32CharArray, 1, 4, fileHandle)) != 4)
	{
		return false;
	}

	if (needsBeToLeConversion)
	{
		outReadUInt32 = swapUInt32Bytes(charArrayToUInt32(uint32CharArray));
		return true;
	}
	else
	{
		outReadUInt32 = charArrayToUInt32(uint32CharArray);
		return true;
	}
}

// RAII classes:

template <class T>
class ArrayGurard
{
private:
	ArrayGurard(const ArrayGurard&);
	ArrayGurard& operator=(const ArrayGurard&);

	T*& arrayRef;
	bool released;

public:
	explicit ArrayGurard(T*& arrayRef) :
			arrayRef(arrayRef),
			released(false)
	{
	}

	~ArrayGurard()
	{
		if (!this->released)
		{
			delete[] this->arrayRef;
		}
	}

	const T* release()
	{
		this->released = true;
		return this->arrayRef;
	}
};

class CloseFileHandleGuard
{
private:
	CloseFileHandleGuard(const CloseFileHandleGuard&);
	CloseFileHandleGuard& operator=(const CloseFileHandleGuard&);

	FILE*& fileHandleRef;

public:
	explicit CloseFileHandleGuard(FILE*& fileHandleRef) :
			fileHandleRef(fileHandleRef)
	{
	}

	~CloseFileHandleGuard()
	{
		if (this->fileHandleRef)
		{
			fclose(this->fileHandleRef);
		}
	}
};

// Helper function to load strings from a .mo file and stores them in a given array

static bool loadMoFileStringsToArray(FILE* moFile,
		uint32_t numberOfStrings,
		uint32_t stringsTableOffsetFromFileBegin,
		bool needsBeToLeConversion,
		std::string* outStringsFromMoFileArray)
{
	if (fseek(moFile, stringsTableOffsetFromFileBegin, SEEK_SET) != 0) return false;

	uint32_t* stringsLengthsArray = NULL;
	ArrayGurard<uint32_t> stringsLengthsArrayGuard(stringsLengthsArray);
	stringsLengthsArray = new uint32_t[numberOfStrings];
	if (!stringsLengthsArray)
	{
		return false;
	}

	uint32_t firstStringOffset;
	uint32_t lastStringOffset;
	{
		uint32_t currentStringLength;
		uint32_t currentStringOffset;
		for (uint32_t i = 0; i < numberOfStrings; i++)
		{
			if (!readUIn32FromFile(moFile, needsBeToLeConversion, currentStringLength)) return false;
			if (!readUIn32FromFile(moFile, needsBeToLeConversion, currentStringOffset)) return false;

			stringsLengthsArray[i] = currentStringLength;

			if (i == 0)
			{
				firstStringOffset = currentStringOffset;
			}
			
			if (i == (numberOfStrings - 1))
			{
				lastStringOffset = currentStringOffset;
			}
		}
	}

	{
		char* stringCharsArray = NULL;
		ArrayGurard<char> stringCharsArrayGuard(stringCharsArray);

		uint32_t stringCharsArraySize = lastStringOffset + stringsLengthsArray[numberOfStrings - 1] + 1 - firstStringOffset;
		if (stringCharsArraySize == 0)
		{
			return false;
		}

		if (fseek(moFile, firstStringOffset, SEEK_SET) != 0) return false;
		stringCharsArray = new char[stringCharsArraySize];
		if (!stringCharsArray)
		{
			return false;
		}
		if (fread(stringCharsArray, 1, stringCharsArraySize, moFile) != stringCharsArraySize) return false;

		const char* stringsCharsArrayIter = stringCharsArray;
		for (uint32_t i = 0; i < numberOfStrings; i++)
		{
			const char* currentStrEndIter = stringsCharsArrayIter + stringsLengthsArray[i] + 1;
			outStringsFromMoFileArray[i] = std::string(stringsCharsArrayIter, currentStrEndIter);
			stringsCharsArrayIter = currentStrEndIter;
		}
	}

	return true;
}


// change local to lang
// below code is copied and modified of tuxpaint/src/i18n.c and tuxpaint/src/i18n.h
enum
{
  LANG_ACH,                     /* Acholi */
  LANG_AF,                      /* Afrikaans */
  LANG_AK,                      /* Akan */
  LANG_AM,                      /* Amharic */
  LANG_AN,                      /* Aragones */
  LANG_AR,                      /* Arabic */
  LANG_AS,                      /* Assamese */
  LANG_AST,                     /* Asturian */
  LANG_AZ,                      /* Azerbaijani */
  LANG_BE,                      /* Belarusian */
  LANG_BG,                      /* Bulgarian */
  LANG_BM,                      /* Bambara */
  LANG_BN,                      /* Bengali */
  LANG_BO,                      /* Tibetan */
  LANG_BR,                      /* Breton */
  LANG_BRX,                     /* Bodo */
  LANG_BS,                      /* Bosnian */
  LANG_CA_VALENCIA,             /* Valencian */
  LANG_CA,                      /* Catalan */
  LANG_CGG,                     /* Kiga */
  LANG_CS,                      /* Czech */
  LANG_CY,                      /* Welsh */
  LANG_DA,                      /* Danish */
  LANG_DE,                      /* German */
  LANG_DOI,                     /* Dogri */
  LANG_EL,                      /* Greek */
  LANG_EN,                      /* English (American) (DEFAULT) */
  LANG_EN_AU,                   /* English (Australian) */
  LANG_EN_CA,                   /* English (Canadian) */
  LANG_EN_GB,                   /* English (British) */
  LANG_EN_ZA,                   /* English (South African) */
  LANG_EO,                      /* Esperanto */
  LANG_ES_MX,                   /* Spanish (Mexican) */
  LANG_ES,                      /* Spanish */
  LANG_ET,                      /* Estonian */
  LANG_EU,                      /* Basque */
  LANG_FA,                      /* Persian */
  LANG_FF,                      /* Fulah */
  LANG_FI,                      /* Finnish */
  LANG_FO,                      /* Faroese */
  LANG_FR,                      /* French */
  LANG_GA,                      /* Irish Gaelic */
  LANG_GD,                      /* Scottish Gaelic */
  LANG_GL,                      /* Galician */
  LANG_GR,                      /* Gronings */
  LANG_GU,                      /* Gujarati */
  LANG_HE,                      /* Hebrew */
  LANG_HI,                      /* Hindi */
  LANG_HR,                      /* Croatian */
  LANG_HU,                      /* Hungarian */
  LANG_HY,                      /* Armenian */
  LANG_I_KLINGON_ROMANIZED,     /* Klingon (Romanized) */
  LANG_ID,                      /* Indonesian */
  LANG_IS,                      /* Icelandic */
  LANG_IT,                      /* Italian */
  LANG_IU,                      /* Inuktitut */
  LANG_JA,                      /* Japanese */
  LANG_KA,                      /* Georgian */
  LANG_KAB,                     /* Kabyle */
  LANG_KN,                      /* Kannada */
  LANG_KM,                      /* Khmer */
  LANG_KOK_ROMAN,               /* Konkani (Roman) */
  LANG_KOK,                     /* Konkani (Devanagari) */
  LANG_KO,                      /* Korean */
  LANG_KS_DEVANAGARI,           /* Kashmiri (Devanagari script) */
  LANG_KS,                      /* Kashmiri (Perso-Arabic script) */
  LANG_KU,                      /* Kurdish */
  LANG_LB,                      /* Luxembourgish */
  LANG_LG,                      /* Luganda */
  LANG_LT,                      /* Lithuanian */
  LANG_LV,                      /* Latvian */
  LANG_MAI,                     /* Maithili */
  LANG_ML,                      /* Malayalam */
  LANG_MK,                      /* Macedonian */
  LANG_MN,                      /* Mongolian */
  LANG_MNI_BENGALI,             /* Manipuri (Bengali script) */
  LANG_MNI_METEI_MAYEK,         /* Manipuri (Metei Mayek script) */
  LANG_MR,                      /* Marath */
  LANG_MS,                      /* Malay */
  LANG_NB,                      /* Norwegian Bokmal */
  LANG_NE,                      /* Nepali */
  LANG_NL,                      /* Dutch */
  LANG_NN,                      /* Norwegian Nynorsk */
  LANG_NR,                      /* Ndebele */
  LANG_NSO,                     /* Northern Sotho */
  LANG_OC,                      /* Occitan */
  LANG_OJ,                      /* Ojibway */
  LANG_OR,                      /* Odia */
  LANG_PA,                      /* Punjabi */
  LANG_PL,                      /* Polish */
  LANG_PT_BR,                   /* Portuguese (Brazilian) */
  LANG_PT,                      /* Portuguese */
  LANG_RO,                      /* Romanian */
  LANG_RU,                      /* Russian */
  LANG_RW,                      /* Kinyarwanda */
  LANG_SAT_OL_CHIKI,
  LANG_SAT,                     /* Santali */
  LANG_SA,                      /* Sanskrit */
  LANG_SC,                      /* Sardinian */
  LANG_SD,                      /* Sindhi (Perso-Arabic) */
  LANG_SD_DEVANAGARI,           /* Sindhi (Devanagari) */
  LANG_SHS,                     /* Shuswap */
  LANG_SI,                      /* Sinhala */
  LANG_SK,                      /* Slovak */
  LANG_SL,                      /* Slovenian */
  LANG_SON,                     /* Songhay */
  LANG_SQ,                      /* Albanian */
  LANG_SR_LATIN,                /* Serbian (latin) */
  LANG_SR,                      /* Serbian (cyrillic) */
  LANG_SU,                      /* Sundanese */
  LANG_SV,                      /* Swedish */
  LANG_SW,                      /* Swahili */
  LANG_TA,                      /* Tamil */
  LANG_TE,                      /* Telugu */
  LANG_TH,                      /* Thai */
  LANG_TL,                      /* Tagalog */
  LANG_TR,                      /* Turkish */
  LANG_TW,                      /* Twi */
  LANG_UK,                      /* Ukrainian */
  LANG_UR,                      /* Urdu */
  LANG_VEC,                     /* Venetian */
  LANG_VE,                      /* Venda */
  LANG_VI,                      /* Vietnamese */
  LANG_WA,                      /* Walloon */
  LANG_WO,                      /* Wolof */
  LANG_XH,                      /* Xhosa */
  LANG_ZAM,                     /* Zapotec (Miahuatlan) */
  LANG_ZH_CN,                   /* Chinese (Simplified) */
  LANG_ZH_TW,                   /* Chinese (Traditional) */
  LANG_ZU,                      /* Zulu */
  NUM_LANGS
};

const char *lang_prefixes[NUM_LANGS] = {
  "ach",
  "af",
  "ak",
  "am",
  "an",
  "ar",
  "as",
  "ast",
  "az",
  "be",
  "bg",
  "bm",
  "bn",
  "bo",
  "br",
  "brx",
  "bs",
  "ca@valencia",
  "ca",
  "cgg",
  "cs",
  "cy",
  "da",
  "de",
  "doi",
  "el",
  "en",
  "en_AU",
  "en_CA",
  "en_GB",
  "en_ZA",
  "eo",
  "es_MX",
  "es",
  "et",
  "eu",
  "fa",
  "ff",
  "fi",
  "fo",
  "fr",
  "ga",
  "gd",
  "gl",
  "gos",
  "gu",
  "he",
  "hi",
  "hr",
  "hu",
  "hy",
  "tlh",
  "id",
  "is",
  "it",
  "iu",
  "ja",
  "ka",
  "kab",
  "kn",
  "km",
  "kok@roman",
  "kok",
  "ko",
  "ks@devanagari",
  "ks",
  "ku",
  "lb",
  "lg",
  "lt",
  "lv",
  "mai",
  "ml",
  "mk",
  "mn",
  "mni",
  "mni@meiteimayek",
  "mr",
  "ms",
  "nb",
  "ne",
  "nl",
  "nn",
  "nr",
  "nso",
  "oc",
  "oj",
  "or",
  "pa",
  "pl",
  "pt_BR",
  "pt",
  "ro",
  "ru",
  "rw",
  "sat@olchiki",
  "sat",
  "sa",
  "sc",
  "sd",
  "sd@devanagari",
  "shs",
  "si",
  "sk",
  "sl",
  "son",
  "sq",
  "sr@latin",
  "sr",
  "su",
  "sv",
  "sw",
  "ta",
  "te",
  "th",
  "tl",
  "tr",
  "tw",
  "uk",
  "ur",
  "vec",
  "ve",
  "vi",
  "wa",
  "wo",
  "xh",
  "zam",
  "zh_CN",
  "zh_TW",
  "zu",
};


static const char* get_lang_from_locale(const char *locale)
{
  char *baseloc = strdup(locale);
  char *dot = strchr(baseloc, '.');
  char *at = strchr(baseloc, '@');
  char *cntrycode = strchr(baseloc, '_');
  char straux[255];
  char *ataux = NULL;
  char *ccodeaux = NULL;
  size_t len_baseloc = 0;
  int found = 0;
  int i = 0;
  int langint = LANG_EN;

  if (!locale)
    return NULL;

  /* Remove the .UTF-8 extension, then
     try to find the full locale including country code and variant,
     if it fails, then try to find the language code plus the variant,
     if it still fails, try to find language and country code without the variant,
     finally scan just the lang part.
     as a last resource reverse the scanning
  */

  if(dot)
    *dot = '\0';

  if (cntrycode)
  {
    ccodeaux = strdup(cntrycode);
    *cntrycode = '\0';
  }

  if (at)
  {
    ataux = strdup(at);
    *at = '\0';

    if(cntrycode)
    {
      /* ll_CC@variant */
      //if (found == 0)  printf("ll_CC@variant check\n");
      snprintf(straux, 255, "%s%s%s", baseloc, ccodeaux, ataux);
      len_baseloc = strlen(straux);
      for (i = 0; i < NUM_LANGS && found == 0; i++)
      {
	// Case-insensitive (both "pt_BR" and "pt_br" work, etc.)
	if (len_baseloc == strlen(lang_prefixes[i]) &&
	    !strncasecmp(straux, lang_prefixes[i], len_baseloc))
	{
	  langint = i;
	  found = 1;
	}
      }
    }

    /* ll@variant*/
    //if (found == 0)  printf("ll@variant check\n");
    snprintf(straux, 255, "%s%s", baseloc, ataux);
    len_baseloc = strlen(straux);
    for (i = 0; i < NUM_LANGS && found == 0; i++)
    {
      // Case-insensitive (both "pt_BR" and "pt_br" work, etc.)
      if (len_baseloc == strlen(lang_prefixes[i]) &&
	  !strncasecmp(straux, lang_prefixes[i], len_baseloc))
      {
	langint = i;
	found = 1;
      }
    }
  }

  if(cntrycode)
    {
      /* ll_CC */
      //if (found == 0)  printf("ll_CC check\n");
      snprintf(straux, 255, "%s%s",baseloc, ccodeaux); 
      len_baseloc = strlen(straux);

      /* Which, if any, of the locales is it? */

      for (i = 0; i < NUM_LANGS && found == 0; i++)
      {
	// Case-insensitive (both "pt_BR" and "pt_br" work, etc.)
	if (len_baseloc == strlen(lang_prefixes[i]) &&
	    !strncasecmp(straux, lang_prefixes[i], strlen(lang_prefixes[i])))
	{
	  langint = i;
	  found = 1;
	}
      }
    }

  /* ll */
  //  if (found == 0)  printf("ll check\n");
  len_baseloc = strlen(baseloc);
  /* Which, if any, of the locales is it? */

  for (i = 0; i < NUM_LANGS && found == 0; i++)
  {
    // Case-insensitive (both "pt_BR" and "pt_br" work, etc.)
    if (len_baseloc == strlen(lang_prefixes[i]) &&
	!strncasecmp(baseloc, lang_prefixes[i], strlen(lang_prefixes[i])))
    {
      langint = i;
      found = 1;
    }
  }

/* Last resource, we should never arrive here, this check depends
   on the right order in lang_prefixes[] 
   Languages sharing the same starting letters must be ordered 
   from longest to shortest, like currently are pt_BR and pt */
// if (found == 0)
  // printf("Language still not found: loc= %s  Trying reverse check as last resource...\n", loc);

  for (i = 0; i < NUM_LANGS && found == 0; i++)
  {
    // Case-insensitive (both "pt_BR" and "pt_br" work, etc.)
    if (!strncasecmp(locale, lang_prefixes[i], strlen(lang_prefixes[i])))
    {
      langint = i;
      found = 1;
    }
  }
  //  printf("langint %i, lang_ext %s\n", langint, lang_prefixes[langint]);

  free(baseloc);
  if (ataux)
    free(ataux);
  if (ccodeaux)
    free(ccodeaux);  
  
  return lang_prefixes[langint];
}

} // namespace internal

} // namespace libintllite

#endif // LIBINTL_INTERNAL_UTIL_HPP_
